--
-- BuyableWheelWeights
-- Specialization for Buyable Twin Weights
--
-- @author  	Manuel Leithner (SFM-Modding)
-- @version 	v2.0
-- @date  		15/10/10
-- @history:	v1.0 - Initial version
--				v2.0 - added network support, changed update to updateTick
-- Edit By Ufolec

BuyableWheelWeights = {};

function BuyableWheelWeights.prerequisitesPresent(specializations)
    return true;
end;

function BuyableWheelWeights:load(xmlFile)

	self.wheelWeightTriggerCallback = BuyableWheelWeights.wheelWeightTriggerCallback;
	self.wheelDelete = BuyableWheelWeights.wheelDelete;
	self.assembleWeights = SpecializationUtil.callSpecializationsFunction("assembleWeights");
	self.disassembleWeights = SpecializationUtil.callSpecializationsFunction("disassembleWeights");

	self.checkString = Utils.getNoNil(getXMLString(xmlFile, "vehicle.wheelWeights#checkString"), "standart");
	self.usingTrigger = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.wheelWeights#activationTrigger"));
	
	
	addTrigger(self.usingTrigger, "wheelWeightTriggerCallback", self);
    self.deleteListenerId = addDeleteListener(self.usingTrigger, "wheelDelete", self);
	
	self.belts = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.wheelWeights#belts"));
	
    self.buyableWeights = {};
    local i = 0;
    while true do
        local wheelnamei = string.format("vehicle.wheelWeights.weight(%d)", i);
        local wheel = {};
        local wheelIndex = getXMLInt(xmlFile, wheelnamei .. "#wheelIndex");
        if wheelIndex == nil then
            break;
        end;
		
		wheel.wheelIndex = wheelIndex;
		wheel.node = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#node"));
		wheel.collision = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#collision"));
		-- collision bug fix: on initial wheel assembling no collision is set
		setRigidBodyType(wheel.collision, "Kinematic");
		setRigidBodyType(wheel.collision, "NoRigidBody");
		wheel.savePosition = Utils.indexToObject(self.components, getXMLString(xmlFile, wheelnamei .. "#savePosition"));
		table.insert(self.buyableWeights, wheel);
		i = i + 1;
	end;

	self.loadedCoords = nil;
	self.wheelWeightsUser = nil;
end;

function BuyableWheelWeights:delete()
	if self.wheelWeightsUser ~= nil then
		self.wheelWeightsUser:onDisassembling(true);
		for _,wheelWeight in pairs(self.buyableWeights) do
			delete(wheelWeight.node);
			delete(wheelWeight.collision);
		end;
		self.buyableWeights = {};
	end;
    removeTrigger(self.usingTrigger);
end;

function BuyableWheelWeights:wheelDelete()
    removeDeleteListener(self.usingTrigger, self.deleteListenerId);
end;

function BuyableWheelWeights:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	if id ~= -1 then
		local wheelWeightsUser = networkGetObject(id);
		if wheelWeightsUser ~= nil then
			self:assembleWeights(wheelWeightsUser, true);
		end;
	end;
end;

function BuyableWheelWeights:writeStream(streamId, connection)
	local idToWrite = -1;
	if self.wheelWeightsUser ~= nil then
		idToWrite = networkGetObjectId(self.wheelWeightsUser);
	end;
	streamWriteInt32(streamId, idToWrite);
end;

function BuyableWheelWeights:mouseEvent(posX, posY, isDown, isUp, button)
end;

function BuyableWheelWeights:keyEvent(unicode, sym, modifier, isDown)
end;

function BuyableWheelWeights:update(dt)
end;

function BuyableWheelWeights:updateTick(dt)

	-- try to load saved twinwheels
	if self.loadedCoords ~= nil then
		for k,steerable in pairs(g_currentMission.steerables) do
			local a,b,c = getWorldTranslation(steerable.rootNode);
			local distance = Utils.vector3Length(self.loadedCoords.x-a, self.loadedCoords.y-b, self.loadedCoords.z-c);
			if distance < 0.15 then				
				self:assembleWeights(steerable);
				break;
			end;
		end;
		self.loadedCoords = nil;
	end;
end;

function BuyableWheelWeights:draw()	
end;

function BuyableWheelWeights:wheelWeightTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	local vehicle = g_currentMission.controlledVehicle;
	if vehicle ~= nil then
		
		if onEnter then		
			if vehicle.rootNode == otherId then
				if vehicle.buyableWheelWeights ~= nil then
					if vehicle.buyableWheelWeights.checkString == self.checkString then
						if self.wheelWeightsUser ~= nil then
							if self.wheelWeightsUser == vehicle then
								vehicle.buyableWheelWeights.wheelsInRange = self;
							end;
						else
							vehicle.buyableWheelWeights.wheelsInRange = self;
						end;
					end;
				end;
			end;
		elseif onLeave then		
			if otherId == vehicle.rootNode then
				if vehicle.buyableWheelWeights ~= nil then
					vehicle.buyableWheelWeights.wheelsInRange = nil;
				end;
			end;
		end;
	end;
end;

function BuyableWheelWeights:assembleWeights(vehicle, noEventSend)
	BuyableWheelWeightsAttachEvent.sendEvent(self, vehicle, noEventSend);

	if self.belts ~= nil then
		setVisibility(self.belts, false);
	end;
	if self.wheelWeightsUser == nil then
		if vehicle.buyableWheelWeights ~= nil then
			self.wheelWeightsUser = vehicle;	
			for k,wheel in pairs(vehicle.wheels) do
				for l,wheelWeight in pairs(self.buyableWeights) do 
					if k == l then	
						unlink(wheelWeight.node);
						unlink(wheelWeight.collision);
						link(wheel.driveNode, wheelWeight.node);	
						link(getParent(wheel.repr), wheelWeight.collision);	
						local x,y,z = getWorldTranslation(wheel.driveNode);
						x,y,z = worldToLocal(getParent(wheelWeight.collision), x,y,z);
						setTranslation(wheelWeight.collision,x,y,z);
						--setRigidBodyType(wheelWeight.collision, "Kinematic");
						break;
					end;
				end;
			end;	
			self.wheelWeightsUser:onAssemblingWeights(self);
		end;
	end;
end;

function BuyableWheelWeights:disassembleWeights(noEventSend)
	BuyableWheelWeightsDetachEvent.sendEvent(self, noEventSend);

	self.wheelWeightsUser:onDisassemblingWeights();

	if self.belts ~= nil then
		setVisibility(self.belts, true);
	end;
	
	for k,wheel in pairs(self.buyableWeights) do
		setRigidBodyType(wheel.collision, "NoRigidBody");
		unlink(wheel.collision);
		unlink(wheel.node);
		setRotation(wheel.collision, 0,0,0);
		setTranslation(wheel.collision, 0,0,0);
		link(wheel.savePosition, wheel.collision);
		link(wheel.collision, wheel.node);
	end;	
	self.wheelWeightsUser = nil;	
end;


function BuyableWheelWeights:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)

	if not resetVehicles then
		local valueStr = getXMLString(xmlFile, key.."#attacherCoords");		
		if valueStr ~= nil then
			local x,y,z = Utils.getVectorFromString(valueStr);
			self.loadedCoords = {x = x,y = y,z = z};			
		end;
	end; 

    return BaseMission.VEHICLE_LOAD_OK;
end;

function BuyableWheelWeights:getSaveAttributesAndNodes(nodeIdent)	
    local attributes = nil;
	
	if self.wheelWeightsUser ~= nil then
		local x,y,z = getWorldTranslation(self.wheelWeightsUser.rootNode);
		attributes = 'attacherCoords="'.. x .. " " .. y .. " " .. z ..'"';
	end;
	
    return attributes, nil;
end;




